//////////////////////////////
// Tile Movement Code v1.42 //
//      By Radnen 2008      //
//////////////////////////////

//////////////////////////////////////////////////////
// You could *attach other NPC's* to this engine so
// it's not only player based. It may even be possible
// to have 2 instances with different inputs as to
// support *mulitple players*.
//////////////////////////////////////////////////////
// EX:
// MyChar = new TileMovementEngine(32);
// HisChar = new TileMovementEngine(32);
// MyChar.attachInput("my_character");
// HisChar.attachInput("his_character");
// HisChar.upKey = KEY_W;
// HisChar.downKey = KEY_S;
// HisChar.leftKey = KEY_A;
// HisChar.rightKey = KEY_D;
/////////////////////////////////////////////////////
// However it's not required to attach other NPC's //
// Only Player Characters because they recieve     //
// Input from the keyboard and trigger events.     //
/////////////////////////////////////////////////////

const METHOD_DIRECTION = 0;
const METHOD_FRAME = 1;

function TileMovementEngine()
{
	if (this instanceof TileMovementEngine == false)
	{
		return new TileMovementEngine();
	}
	
	this.initialize = true;
	this.tileSize = 0;
	this.tileMoveSize = 0;
	this.paused = false;
	this.isMoving = false;
	this.lastX = 0;
	this.lastY = 0;
	this.postMovement = false;
	this.upKey = KEY_UP;
	this.downKey = KEY_DOWN;
	this.leftKey = KEY_LEFT;
	this.rightKey = KEY_RIGHT;
	this.moveTile = false;
	this.input = null;
	this.reposition = true;
	this.direction = "";
	this.command;
	this.commands = [];
	this.commands["north"] = COMMAND_MOVE_NORTH;
	this.commands["south"] = COMMAND_MOVE_SOUTH;
	this.commands["east"] = COMMAND_MOVE_EAST;
	this.commands["west"] = COMMAND_MOVE_WEST;
	this.faceDirection = "";
	this.defaultFrame = 0;
	this.directionMethod = METHOD_DIRECTION;
	this.face;
	this.faces = [];
	this.faces["north"] = COMMAND_FACE_NORTH;
	this.faces["south"] = COMMAND_FACE_SOUTH;
	this.faces["east"] = COMMAND_FACE_EAST;
	this.faces["west"] = COMMAND_FACE_WEST;
	// User set Tile Actions:
	this.onTileLeave = function() { };
	this.onTileEnter = function() { };
	this.onTile = function() { };
	
	// To get rid of AttachPlayer() input control:
	BindKey(KEY_UP   , '', '');
	BindKey(KEY_DOWN , '', '');
	BindKey(KEY_LEFT , '', '');
	BindKey(KEY_RIGHT, '', '');
}

/////////////////   THIS GOES IN AN UPDATE SCRIPT   //////////////////////

TileMovementEngine.prototype.update = function()
{
	// This is called right after it's first update to set tile size, based on the map.
	if(this.initialize)
	{
		this.setTileSize();
		this.initialize = false;
	}
	
	this.onTile(); // User set onTile function. Stuff here will continously get updated.
	
	this.inputX = GetPersonX(this.input);
	this.inputY = GetPersonY(this.input);
	
	// This bit will process movement if movement is detected:
	if (this.isMoving)
	{
		this.lastX = this.inputX;
		this.lastY = this.inputY;
		this.onTileLeave(); // User set onTileLeave function.
		if (this.moveTile)
		{
			QueuePersonCommand(this.input, this.face, true);
			for (var i = 0; i < this.tileMoveSize; ++i)
			{
				QueuePersonCommand(this.input, this.command, false);
			}
		}
		// Set facing to a predefined face or frame, to act as a "stop" frame
		else if (this.directionMethod == 0) SetPersonDirection(this.input, this.faceDirection + this.direction);
		else if (this.directionMethod == 1) SetPersonFrame(this.input, this.defaultFrame);
		this.isMoving = false;
		this.postMovement = true;
	}
		
	// Post Movement / Idle:
	if (IsCommandQueueEmpty(this.input))
	{
		// This will feed keyboard input into the move command
		if (IsKeyPressed(this.upKey))    this.move("north");
		if (IsKeyPressed(this.downKey))  this.move("south");
		if (IsKeyPressed(this.leftKey))  this.move("west");
		if (IsKeyPressed(this.rightKey)) this.move("east");
	
		if (this.postMovement)
		{
			// Set facing to a predefined face or frame, to act as a "stop" frame
			if (this.directionMethod == 0) SetPersonDirection(this.input, this.faceDirection + this.direction);
			if (this.directionMethod == 1) SetPersonFrame(this.input, this.defaultFrame);
			
			// Do repositioning if player is off of the tiles center
			if (this.reposition)
			{
				var XDelta = Math.pow(this.inputX - this.lastX, 2) / this.tileSize;
				var YDelta = Math.pow(this.inputY - this.lastY, 2) / this.tileSize;
				if (XDelta != this.tileSize && XDelta != 0) SetPersonX(this.input, this.lastX);
				if (YDelta != this.tileSize && YDelta != 0) SetPersonY(this.input, this.lastY);
			}
					
			this.onTileEnter(); // User set OnTileEnter function.
			
		  /*This next bit is used to check for if a player landed on a trigger; normally
			Sphere would have checked this on it's own. But that wasn't for a
			tile based engine. On this system the trigger would only execute after the entity
			moved a tile, so this cures the trigger firing too soon problem.*/
			if(IsTriggerAt(this.inputX, this.inputY, 0))
			{
				ExecuteTrigger(this.inputX, this.inputY, 0);
			}
			
			this.postMovement = false;
		}
	}
}

//////////////////////////        INPUT        ///////////////////////////////////

// This is used to move the entity an x amount of squares in any given direction.
// There was no need to set up a facing direction; it'll know.
TileMovementEngine.prototype.move = function(direction)
{
	if (!this.paused)
	{
		// Check to see if player can move this direction:
		if(!this.isObstructedAt(this.commands[direction])) this.moveTile = true;
		else this.moveTile = false;
		
		this.direction = direction;
		this.command = this.commands[direction];
		this.face = this.faces[direction];
		this.isMoving = true;
	}
}

// This will simulate a person who is pushed; so his facing doesn't change.
// Be creative and see if you can make the hero push other entities :)
// Please note that the entity being pushed needs to be a person of the engine.
TileMovementEngine.prototype.push = function(direction, tiles)
{
	if(!this.paused)
	{
		this.command = this.commands[direction];
		this.tiles = tiles;
		this.isMoving = true;
	}
}

// Use this for the hero or player character.
TileMovementEngine.prototype.attachInput = function(name)
{
	this.paused = false; // allow keyboard input
	this.input = name; // point to this person.
}

// Use this to detach the player input //
TileMovementEngine.prototype.detachInput = function()
{
	this.paused = true;
}

// Use this to check if the input is attached //
TileMovementEngine.prototype.isInputAttached = function()
{
	if (this.input.length == 0) return false;
	else return true;
}

// This will return the input person, if one exists //
TileMovementEngine.prototype.getInputPerson = function()
{
	if (this.isInputAttached()) return this.input;
	else throw "No input attached to engine.";
}

// Fairly simple function, returns if all of the tiles that were queued up
// have been emptied. Useful for events that'll happen at the end of a path
TileMovementEngine.prototype.areCommandsClear = function()
{
	if (this.isInputAttached()) return IsCommandQueueEmpty(this.input);
	else throw "No input attached to engine.";
}

// Will pause all movement, yet keep the input //
TileMovementEngine.prototype.pauseEngine = function()
{
	this.paused = true;
}

// Will continue movement //
TileMovementEngine.prototype.unpauseEngine = function()
{
	this.paused = false;
}

// Will return true if the engine isn't paused //
TileMovementEngine.prototype.isPaused = function()
{
	return this.paused;
}

// This will force stop the engine, use it for abrupt stopping, mid-tile even.
// Restore will instead of stopping mid-tile, the object will be restored to it's last position.
TileMovementEngine.prototype.stopEngine = function(restore)
{
	this.isMoving = false;
	ClearPersonCommands(this.input);
	if (restore) SetPersonXYFloat(this.input, this.lastX, this.lastY);
}

// Is used within the move method and will check valid obstruction zones
// up to a tile away from the input character.
TileMovementEngine.prototype.isObstructedAt = function(direction)
{
	const X = this.inputX;
	const Y = this.inputY;
	const PLAYER = this.input;
	const TILED2 = this.tileSize/2;
	
	function ObstructedAt(x, y)
	{
		return IsPersonObstructed(PLAYER, x, y);
	}
	
	switch(direction)
	{
		case "northeast": if (ObstructedAt(X + TILED2, Y - TILED2)) return true; break;
		case "northwest": if (ObstructedAt(X - TILED2, Y - TILED2)) return true; break;
		case "southeast": if (ObstructedAt(X + TILED2, Y + TILED2)) return true; break;
		case "southwest": if (ObstructedAt(X - TILED2, Y + TILED2)) return true; break;
		case "north": if (ObstructedAt(X, Y - TILED2)) return true; break;
		case "south": if (ObstructedAt(X, Y + TILED2)) return true; break;
		case "east":  if (ObstructedAt(X + TILED2, Y)) return true; break;
		case "west":  if (ObstructedAt(X - TILED2, Y)) return true; break;
		default: return false; break;
	}
}

// Will set the tile size and dynamically shift it according to the input players speed //
TileMovementEngine.prototype.setTileSize = function()
{
	this.tileSize = GetTileWidth()
	this.tileMoveSize = this.tileSize / GetPersonSpeedX(this.input);
}